/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#ifndef _SURFACE_H_
#define _SURFACE_H_

#include <stdlib.h>
#include <pthread.h>
#include <sys/time.h>

#include "GraphicalSurface.h"
#include "PlatformSurface.h"
#include "PixelFormat.h"
#include "InputManager.h"
#include "ViewportTransform.h"
#include "Log.h"
#include <string.h>

typedef std::list <std::string *> SeatList;
/**
 * Represents a graphical surface generated by an application. Always contained in layers.
 */
class Surface : public GraphicalSurface
{
public:
    bool m_deferredSrcRegionSet;
    bool m_deferredDstRegionSet;
    int m_TextureUpdateCounter;
    bool m_opaqueRegion;

    PixelFormat getPixelFormat() const
    {
        return m_pixformat;
    }

    void setPixelFormat(PixelFormat pf)
    {
        m_pixformat = pf;
    }

    unsigned int getContainingLayerId() const
    {
        return m_layerId;
    }

    void setContainingLayerId(unsigned int id)
    {
        m_layerId = id;
    }

    bool hasNativeContent() const
    {
        return m_hasNativeContent;
    }

    void removeNativeContent()
    {
        m_hasNativeContent = false;
    }

    void setNativeContent(long nativeHandle)
    {
        if (!hasNativeContent())
        {
            m_nativeHandle = nativeHandle;
            m_hasNativeContent = true;
        }
    }

    long getNativeContent() const
    {
        return hasNativeContent() ? m_nativeHandle : -1;
    }

    void setNativeBuffer(void *buffer)
    {
        m_nativeBuffer = buffer;
    }

    void *getNativeBuffer() const
    {
        return m_nativeBuffer;
    }


    bool hasNativeBuffer() const
    {
        return m_nativeBuffer != 0;
    }

    void calculateTargetDestination(const Rectangle &layerSource, const Rectangle &layerDestination)
    {
        if (ViewportTransform::isFullyCropped(getDestinationRegion(), layerSource))
        {
            m_isCropped = true;
            return; // skip rendering of this surface, because it is cropped by layer source region
        }
        else
        {
            m_isCropped = false;
        }

        const FloatRectangle layerSourceRegion = layerSource;
        const FloatRectangle layerDestinationRegion = layerDestination;

        FloatRectangle targetSource = getSourceRegion();
        FloatRectangle targetDestination = getDestinationRegion();

        ViewportTransform::applyLayerSource(layerSourceRegion, targetSource, targetDestination);
        ViewportTransform::applyLayerDestination(layerDestinationRegion, layerSourceRegion, targetDestination);
        m_targetDestination = targetDestination;
        m_targetSource = targetSource;
    }

    bool isCropped()
    {
        return m_isCropped;
    }

    const FloatRectangle& getTargetSourceRegion() const
    {
        return m_targetSource;
    }

    const FloatRectangle& getTargetDestinationRegion() const
    {
        return m_targetDestination;
    }

    /**
     * Indicate from which input devices the Surface can accept events.
     *
     * \param[in] devices Bit masks of supported devices
     * \param[in] accept Indicate if the surface should accept or reject input events from the specified list of devices
     */
    void updateInputEventAcceptanceFrom(InputDevice devices, bool accept)
    {
        if (accept)
        {
            pthread_mutex_lock(&m_inputAcceptMutex);
            m_acceptInputFrom = (InputDevice) (m_acceptInputFrom | devices);
            pthread_mutex_unlock(&m_inputAcceptMutex);
        }
        else
        {
            pthread_mutex_lock(&m_inputAcceptMutex);
            m_acceptInputFrom = (InputDevice) (m_acceptInputFrom  & ~devices);
            pthread_mutex_unlock(&m_inputAcceptMutex);
        }
    }

    /**
     * Indicate if the surface accepts input events from the specified list of devices.
     *
     * \param[in] devices Bit masks of devices
     * \return true if input events are accepted for ALL of the specified devices
     * \return false if input events are not accepted for at least ONE of the specified devices
     */
    bool isInputEventAcceptedFrom(InputDevice devices)
    {
        bool ret;

        pthread_mutex_lock(&m_inputAcceptMutex);
        ret = ((m_acceptInputFrom & devices) == devices);
        pthread_mutex_unlock(&m_inputAcceptMutex);

        return ret;
    }

    /**
     * Get the set of devices from which the surface can accept input events
     *
     * \return Bitmask of InputDevice
     */
    InputDevice getInputEventAcceptanceOnDevices() const
    {
        return m_acceptInputFrom;
    }

    void setCorrectedSrcRegion(const FloatRectangle& region)
    {
    	m_correctedSrcRegion = region;
    }

    const FloatRectangle& getCorrectedSrcRegion() const
    {
    	return m_correctedSrcRegion;
    }

    void setSourceRegionDeferred(const Rectangle& region)
    {
    	m_srcRegionDeferred = region;
    	m_deferredSrcRegionSet = true;
    }

    const Rectangle& getSourceRegionDeferred() const
    {
    	return m_srcRegionDeferred;
    }

    void setDestinationRegionDeferred(const Rectangle& region)
    {
    	m_dstRegionDeferred = region;
    	m_deferredDstRegionSet = true;
    }

    const Rectangle& getDestinationRegionDeferred() const
    {
    	return m_dstRegionDeferred;
    }

    /**
     * Platform specific Object containing surface information specific to a used platform.
     * This typically contains a native window handle/surface handle or information only used
     * by a specific renderer.unsigned
     */
    PlatformSurface *platform; // platform/rendering specific window attributes

    int updateCounter;
    int fpsUpdateCounter;
    int frameCounter;
    int fpsFrameCounter;
    float surfaceFps;
    float surfaceFrameFps;
    struct timeval lastFrameTime;
    struct timeval lastUpdateTime;

    Surface(int creatorPid)
    : GraphicalSurface(TypeSurface, creatorPid)
    , m_deferredSrcRegionSet(false)
    , m_deferredDstRegionSet(false)
    ,m_TextureUpdateCounter(0)
    , m_opaqueRegion(false)
    , platform(NULL)
    , updateCounter(0)
    , fpsUpdateCounter(0)
    , frameCounter(0)
    , fpsFrameCounter(0)
    , surfaceFps(0.0)
    , surfaceFrameFps(0.0)
    , lastFrameTime()
    , lastUpdateTime()
    , m_nativeHandle(0)
    , m_pixformat(PIXELFORMAT_UNKNOWN)
    , m_layerId(INVALID_ID)
    , m_hasNativeContent(false)
    , m_acceptInputFrom(INPUT_DEVICE_ALL)
    , m_isCropped(false)
    , m_targetDestination(0.0, 0.0, 0.0, 0.0)
    , m_targetSource(0.0, 0.0, 0.0, 0.0)
    , m_correctedSrcRegion(0.0, 0.0, 0.0, 0.0)
    , m_srcRegionDeferred (0, 0, 0, 0)
    , m_dstRegionDeferred(0, 0, 0, 0)
    , m_nativeBuffer(0)
    , m_pAcceptedSeatList()
    , m_FocusBitMask (0)
    {
        std::string *pDefaultSeat = new std::string("seat0");
        if (NULL != pDefaultSeat)
        {
            m_pAcceptedSeatList.push_back(pDefaultSeat);
        }
        else
        {
            LOG_ERROR("Surface", "Failed to created default seat acceptance");
        }
        pthread_mutex_init(&m_inputAcceptMutex, NULL);
    }

    Surface(int id, int creatorPid)
    : GraphicalSurface(id, TypeSurface, creatorPid)
    , m_deferredSrcRegionSet(false)
    , m_deferredDstRegionSet(false)
    ,m_TextureUpdateCounter(0)
    , m_opaqueRegion(false)
    , platform(NULL)
    , updateCounter(0)
    , fpsUpdateCounter(0)
    , frameCounter(0)
    , fpsFrameCounter(0)
    , surfaceFps(0.0)
    , surfaceFrameFps(0.0)
    , lastFrameTime()
    , lastUpdateTime()
    , m_nativeHandle(0)
    , m_pixformat(PIXELFORMAT_UNKNOWN)
    , m_layerId(INVALID_ID)
    , m_hasNativeContent(false)
    , m_acceptInputFrom(INPUT_DEVICE_ALL)
    , m_isCropped(false)
    , m_targetDestination(0.0, 0.0, 0.0, 0.0)
    , m_targetSource(0.0, 0.0, 0.0, 0.0)
    , m_correctedSrcRegion(0.0, 0.0, 0.0, 0.0)
    , m_srcRegionDeferred (0, 0, 0, 0)
    , m_dstRegionDeferred(0, 0, 0, 0)
    , m_nativeBuffer(0)
    , m_pAcceptedSeatList()
    , m_FocusBitMask (0)
    {
        //Always keep seat0 to NOT disturb current applications
        std::string *pDefaultSeat = new std::string("seat0");
        if (NULL != pDefaultSeat)
        {
            m_pAcceptedSeatList.push_back(pDefaultSeat);
        }
        else
        {
            LOG_ERROR("Surface", "Failed to created default seat acceptance");
        }
        pthread_mutex_init(&m_inputAcceptMutex, NULL);
    }
    bool AddSeat(char* pSeatName)
    {
        SeatList::iterator it;
        bool seatAdditionSuccess = true;
        bool seatExists = false;
        if (NULL != pSeatName)
        {
            if (strlen(pSeatName) > 0)
            {
                for (it = m_pAcceptedSeatList.begin(); it != m_pAcceptedSeatList.end(); it++)
                {
                    if (0 == (*it)->compare(pSeatName))
                    {
                        seatExists = true;
                    }
                }
                if (false == seatExists)
                {
                    std::string *pSeat = new std::string(pSeatName);
                    if (NULL != pSeat)
                    {
                        m_pAcceptedSeatList.push_back(pSeat);
                    }
                    else
                    {
                        LOG_ERROR("Surface", "No memory to update the seat name");
                        seatAdditionSuccess = false;
                    }
                }
            }
        }
        return seatAdditionSuccess;
    }

    bool setAcceptedSeats(char **pSeatArray, unsigned int arraySize)
    {
        unsigned int seatCount;
        bool seatUpdateSuccess = true;
        SeatList removeList;
        SeatList::iterator it;
        //remove seats which are not present in application provided seat array
        getRemoveSeatList(&removeList, pSeatArray, arraySize);
        if ((NULL != pSeatArray) && (0 < arraySize))
        {
            for (seatCount = 0; (seatCount < arraySize)
                              && seatUpdateSuccess; seatCount++)
            {
                seatUpdateSuccess &= AddSeat(pSeatArray[seatCount]);
            }
        }
        for (it = removeList.begin(); it != removeList.end(); it++)
        {
            m_pAcceptedSeatList.remove(*it);
            delete (*it);
        }
        return seatUpdateSuccess;
    }

    void getRemoveSeatList(SeatList *pSeatList, char **pSeatArray, unsigned int arraySize)
    {
        unsigned int seatCount;
        SeatList::iterator it;
        //remove seats which are not present in application provided seat array
        if ((NULL == pSeatArray) || (0 == arraySize))
        {
            *pSeatList = m_pAcceptedSeatList;
        }
        else
        {
            for (it = m_pAcceptedSeatList.begin(); it != m_pAcceptedSeatList.end(); it++)
            {
                for (seatCount = 0; (seatCount < arraySize); seatCount++)
                {
                    if ((NULL != pSeatArray[seatCount]) && (strlen(pSeatArray[seatCount]) > 0 ))
                    {
                        if(0 == (*it)->compare(pSeatArray[seatCount]))
                        {
                            break;
                        }
                    }
                }

                if (seatCount >= arraySize)
                {
                    /*Seat is not present in the seat array to be set*/
                    pSeatList->push_back(*it);
                }
            }
        }
    }

    SeatList& getAcceptedSeats(void)
    {
        return m_pAcceptedSeatList;
    }

    bool getAcceptedSeats(char*** pSeatArray, unsigned int *num_seats)
    {
        SeatList::iterator it;
        bool successGetSeats = false;
        unsigned seatIndex = 0;
        if ((NULL != pSeatArray) && (NULL != num_seats))
        {
            *pSeatArray = (char **)malloc(sizeof(char*) * m_pAcceptedSeatList.size());
            if (NULL != *pSeatArray)
            {
                successGetSeats = true;
                for (it = m_pAcceptedSeatList.begin(); it != m_pAcceptedSeatList.end()
                    && (true == successGetSeats); it++)
                {
                    (*pSeatArray)[seatIndex] = strdup((*it)->c_str());
                    if (NULL == (*pSeatArray)[seatIndex])
                    {
                        successGetSeats = false;
                        LOG_ERROR("Surface", "Not enough memory to allocate seat string");
                    }
                    else
                    {
                        seatIndex++;
                    }
                }
            }
            *num_seats = seatIndex;
        }
        return successGetSeats;
    }
    bool checkSeatAcceptance(char *pSeatName)
    {
        SeatList::iterator it;
        bool acceptance = false;
        for (it = m_pAcceptedSeatList.begin(); it != m_pAcceptedSeatList.end()
            && (false == acceptance); it++)
        {
            if (0 == (*it)->compare(pSeatName))
            {
                acceptance = true;
            }
        }
        return acceptance;
    }
    void setFocusBitmap(ilmInputDevice bitmap)
    {
        m_FocusBitMask |= bitmap;
    }
    ilmInputDevice getFocusBitmap(void)
    {
        return m_FocusBitMask;
    }
    void resetFocusBitmap(ilmInputDevice bitmap)
    {
        m_FocusBitMask &= ~bitmap;
    }
    ~Surface()
    {
        SeatList::iterator it;
        for (it = m_pAcceptedSeatList.begin(); it != m_pAcceptedSeatList.end(); it++)
        {
            delete (*it);
        }
        m_pAcceptedSeatList.clear();
        pthread_mutex_destroy(&m_inputAcceptMutex);
    }


private:
    long m_nativeHandle;
    PixelFormat m_pixformat;
    unsigned int m_layerId;
    bool m_hasNativeContent;
    /** \brief Bitmask to represent from which kind of devices the surface can accept
      * input events. By default, all devices are accepted. */
    InputDevice m_acceptInputFrom;
    /** \brief m_acceptInputFrom can be accessed concurently. To make it thread safe, we need a mutex */
    pthread_mutex_t m_inputAcceptMutex;
    bool m_isCropped;
    FloatRectangle m_targetDestination;
    FloatRectangle m_targetSource;
    FloatRectangle m_correctedSrcRegion;
    Rectangle m_srcRegionDeferred;
    Rectangle m_dstRegionDeferred;
    void *m_nativeBuffer;
    SeatList m_pAcceptedSeatList;
    ilmInputDevice m_FocusBitMask;
};

#endif /* _SURFACE_H_ */

